#!/usr/bin/env python3
"""
WhatsApp Web Message Sender
A simple script to send messages via WhatsApp Web
"""

import time
import sys
import os
from selenium import webdriver
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.chrome.service import Service
from webdriver_manager.chrome import ChromeDriverManager
import logging

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

def setup_driver():
    """Set up Chrome driver with appropriate options"""
    chrome_options = Options()
    chrome_options.add_argument("--no-sandbox")
    chrome_options.add_argument("--disable-dev-shm-usage")
    chrome_options.add_argument("--disable-blink-features=AutomationControlled")
    chrome_options.add_experimental_option("excludeSwitches", ["enable-automation"])
    chrome_options.add_experimental_option('useAutomationExtension', False)
    
    try:
        # Install ChromeDriver automatically
        service = Service(ChromeDriverManager().install())
        driver = webdriver.Chrome(service=service, options=chrome_options)
        driver.execute_script("Object.defineProperty(navigator, 'webdriver', {get: () => undefined})")
        return driver
    except Exception as e:
        logger.error(f"Failed to setup driver: {e}")
        return None

def send_whatsapp_message(number, message):
    """Send message to WhatsApp contact"""
    driver = setup_driver()
    
    if not driver:
        print("❌ Failed to setup Chrome driver. Please make sure Chrome is installed.")
        return False
    
    try:
        logger.info("🌐 Opening WhatsApp Web...")
        driver.get("https://web.whatsapp.com")
        
        print("\n📱 Instructions:")
        print("1. Look for the QR code that appears on the screen")
        print("2. Open WhatsApp on your phone")
        print("3. Go to Settings > WhatsApp Web")
        print("4. Scan the QR code with your phone")
        print("\n⏳ Waiting for WhatsApp Web to load...")
        time.sleep(5)
        
        # Wait for QR code or chats
        try:
            # Check if already logged in
            WebDriverWait(driver, 10).until(
                EC.presence_of_element_located((By.CSS_SELECTOR, "[data-testid='chat-list']"))
            )
            print("✅ Already logged in!")
        except:
            print("\n📱 Please scan the QR code with your phone's WhatsApp app")
            print("⏳ Waiting for QR scan... (Will check every 10 seconds)")
            
            # Wait for QR scan (up to 5 minutes)
            for i in range(30):  # 5 minutes
                time.sleep(10)
                try:
                    WebDriverWait(driver, 2).until(
                        EC.presence_of_element_located((By.CSS_SELECTOR, "[data-testid='chat-list']"))
                    )
                    print("✅ QR scanned! Logged in successfully!")
                    break
                except:
                    if i % 6 == 0:  # Every minute
                        minutes = (i * 10) // 60
                        seconds = (i * 10) % 60
                        print(f"⏳ Still waiting for QR scan... ({minutes}m {seconds}s)")
        
        # Search for the contact
        logger.info(f"🔍 Searching for contact: {number}")
        
        # Find search box
        try:
            search_box = WebDriverWait(driver, 10).until(
                EC.element_to_be_clickable((By.CSS_SELECTOR, "[data-testid='chat-list-search']"))
            )
            search_box.click()
            search_box.clear()
            search_box.send_keys(number)
            time.sleep(2)
        except:
            print("❌ Could not find search box")
            return False
        
        # Try to find and click the contact
        try:
            contact = WebDriverWait(driver, 5).until(
                EC.element_to_be_clickable((By.XPATH, f"//span[@title='{number}']"))
            )
            contact.click()
            print("✅ Found contact!")
        except:
            print("📝 Contact not found in search results.")
            print("🔄 Opening first available chat instead...")
            try:
                first_chat = driver.find_element(By.CSS_SELECTOR, "[data-testid='chat-list'] div[role='button']")
                first_chat.click()
                print("ℹ️ Opened first chat - you may need to change the recipient")
            except:
                print("❌ Could not find any chat to open")
                return False
        
        time.sleep(3)
        
        # Send the message
        logger.info(f"📤 Sending message: '{message}'")
        try:
            # Find message input
            message_input = WebDriverWait(driver, 10).until(
                EC.element_to_be_clickable((By.CSS_SELECTOR, "[data-testid='message-composer']"))
            )
            message_input.click()
            message_input.send_keys(message)
            time.sleep(1)
            
            # Send the message
            send_button = driver.find_element(By.CSS_SELECTOR, "[data-testid='send-button']")
            send_button.click()
            
            print(f"🎉 SUCCESS: Message sent to {number}!")
            return True
            
        except Exception as e:
            print(f"❌ Failed to send message: {str(e)}")
            return False
    
    except Exception as e:
        logger.error(f"Error during automation: {str(e)}")
        return False
    
    finally:
        print("\n🔚 Keeping browser open for 15 seconds...")
        time.sleep(15)
        driver.quit()

def main():
    """Main function"""
    # Get user input
    target_number = input("Enter phone number (e.g., 03147362343): ").strip()
    message = input("Enter message: ").strip()
    
    if not target_number or not message:
        print("❌ Both phone number and message are required!")
        sys.exit(1)
    
    print("\n" + "="*50)
    print("🚀 WhatsApp Web Message Sender")
    print("="*50)
    print(f"📱 Target: {target_number}")
    print(f"💬 Message: {message}")
    print("\n⚠️  Requirements:")
    print("- Google Chrome browser installed")
    print("- WhatsApp app on your phone")
    print("\nStarting automation in 5 seconds...")
    time.sleep(5)
    
    success = send_whatsapp_message(target_number, message)
    
    if success:
        print("\n✅ SUCCESS: Message sent successfully!")
    else:
        print("\n❌ FAILED: Could not send message.")
        print("💡 Check the browser window for any error messages.")

if __name__ == "__main__":
    main()